import { MinecraftBlockTypes } from 'mojang-minecraft';
import { Token } from './extern/tokenizr.js';
import { tokenize, throwTokenError, mergeTokens, parseBlock, parseBlockStates, processOps } from './parser.js';
export class Pattern {
    constructor(pattern = '') {
        this.stringObj = '';
        if (pattern) {
            const obj = Pattern.parseArgs([pattern]).result;
            this.block = obj.block;
            this.stringObj = obj.stringObj;
            this.compile();
        }
    }
    /**
     * Sets a block at a location in a dimension.
     * @param loc
     * @param dimension
     * @returns True if the block at the location changed; false otherwise
     */
    setBlock(loc, dimension) {
        try {
            const oldBlock = dimension.getBlock(loc).permutation;
            this.compiledFunc(patternContext, loc, dimension);
            const newBlock = dimension.getBlock(loc).permutation;
            if (oldBlock.type.id != newBlock.type.id) {
                return true;
            }
            for (const state of oldBlock.getAllProperties()) {
                if (state.value != newBlock.getProperty(state.name).value) {
                    return true;
                }
            }
            return false;
        }
        catch (err) {
            //contentLog.error(err);
            return false;
        }
    }
    clear() {
        this.block = null;
        this.stringObj = '';
        this.compiledFunc = null;
    }
    empty() {
        return this.block == null;
    }
    addBlock(block) {
        const states = new Map();
        block.getAllProperties().forEach((state) => {
            if (!state.name.startsWith('wall_connection_type') && !state.name.startsWith('liquid_depth')) {
                states.set(state.name, state.value);
            }
        });
        if (this.block == null) {
            this.block = new ChainPattern(null);
        }
        this.block.nodes.push(new BlockPattern(null, {
            id: block.type.id,
            data: -1,
            states: states
        }));
        this.stringObj = '(picked)';
        this.compile();
    }
    getBlockSummary() {
        let text = '';
        let blockMap = new Map();
        for (const pattern of this.block.nodes) {
            let sub = pattern.block.id.replace('minecraft:', '');
            for (const state of pattern.block.states) {
                const val = state[1];
                if (typeof val == 'string' && val != 'x' && val != 'y' && val != 'z') {
                    sub += `(${val})`;
                    break;
                }
            }
            if (blockMap.has(sub)) {
                blockMap.set(sub, blockMap.get(sub) + 1);
            }
            else {
                blockMap.set(sub, 1);
            }
        }
        let i = 0;
        for (const block of blockMap) {
            if (block[1] > 1) {
                text += `${block[1]}x ${block[0]}`;
            }
            else {
                text += block[0];
            }
            if (i < blockMap.size - 1)
                text += ', ';
            i++;
        }
        return text;
    }
    getPatternInCommand() {
        let blockData;
        if (this.block instanceof BlockPattern) {
            blockData = this.block.block;
        }
        else if (this.block instanceof ChainPattern && this.block.nodes.length == 1 && this.block.nodes[0] instanceof BlockPattern) {
            blockData = this.block.nodes[0].block;
        }
        if (!blockData)
            return;
        let command = blockData.id;
        if (blockData.data != -1) {
            command += ' ' + blockData.data;
        }
        else if (blockData.states?.size) {
            command += '[';
            let i = 0;
            for (const [state, val] of blockData.states.entries()) {
                command += `"${state}":`;
                command += typeof val == 'string' ? `"${val}"` : `${val}`;
                if (i++ < blockData.states.size - 1) {
                    command += ',';
                }
            }
            command += ']';
        }
        return command;
    }
    compile() {
        // contentLog.debug('compiling', this.stringObj, 'to', this.block.compile());
        if (this.block) {
            this.compiledFunc = new Function('ctx', 'loc', 'dim', this.block.compile());
        }
    }
    static parseArgs(args, index = 0) {
        const input = args[index];
        if (!input) {
            return { result: new Pattern(), argIndex: index + 1 };
        }
        const tokens = tokenize(input);
        let token;
        function processTokens(inBracket) {
            let ops = [];
            let out = [];
            const start = tokens.curr();
            function nodeToken() {
                return mergeTokens(token, tokens.curr(), input);
            }
            while (token = tokens.next()) {
                if (token.type == 'id') {
                    out.push(new BlockPattern(nodeToken(), parseBlock(tokens, input, false)));
                }
                else if (token.type == 'number') {
                    const num = token;
                    const t = tokens.next();
                    if (t.value == '%') {
                        processOps(out, ops, new PercentPattern(nodeToken(), num.value));
                    }
                    else {
                        throwTokenError(t);
                    }
                }
                else if (token.value == ',') {
                    processOps(out, ops, new ChainPattern(token));
                }
                else if (token.value == '^') {
                    const t = tokens.next();
                    if (t.type == 'id') {
                        out.push(new TypePattern(nodeToken(), parseBlock(tokens, input, true)));
                    }
                    else if (t.value == '[') {
                        out.push(new StatePattern(nodeToken(), parseBlockStates(tokens)));
                    }
                    else {
                        throwTokenError(t);
                    }
                }
                else if (token.value == '*') {
                    const t = tokens.next();
                    if (t.type != 'id') {
                        throwTokenError(t);
                    }
                    out.push(new RandStatePattern(nodeToken(), parseBlock(tokens, input, true)));
                }
                else if (token.type == 'bracket') {
                    if (token.value == '(') {
                        out.push(processTokens(true));
                    }
                    else if (token.value == ')') {
                        if (!inBracket) {
                            throwTokenError(token);
                        }
                        else {
                            processOps(out, ops);
                            break;
                        }
                    }
                    else {
                        throwTokenError(token);
                    }
                }
                else if (token.type == 'EOF') {
                    if (inBracket) {
                        throwTokenError(token);
                    }
                    else {
                        processOps(out, ops);
                    }
                }
                else {
                    throwTokenError(token);
                }
            }
            if (out.length > 1) {
                throwTokenError(out.slice(-1)[0].token);
            }
            else if (!out.length) {
                throwTokenError(start);
            }
            else if (ops.length) {
                const op = ops.slice(-1)[0];
                throwTokenError(op instanceof Token ? op : op.token);
            }
            return out[0];
        }
        let out;
        try {
            out = processTokens(false);
            out.postProcess();
        }
        catch (error) {
            if (error.pos != undefined) {
                const err = {
                    isSyntaxError: true,
                    idx: index,
                    start: error.pos,
                    end: error.pos + 1,
                    stack: error.stack
                };
                throw err;
            }
            throw error;
        }
        const pattern = new Pattern();
        pattern.stringObj = args[index];
        pattern.block = out;
        pattern.compile();
        return { result: pattern, argIndex: index + 1 };
    }
    static clone(original) {
        const pattern = new Pattern();
        pattern.block = original.block;
        pattern.stringObj = original.stringObj;
        pattern.compile();
        return pattern;
    }
    toString() {
        return `[pattern: ${this.stringObj}]`;
    }
}
class PatternNode {
    constructor(token) {
        this.token = token;
        this.nodes = [];
    }
    postProcess() { }
}
class BlockPattern extends PatternNode {
    constructor(token, block) {
        super(token);
        this.block = block;
        this.prec = -1;
        this.opCount = 0;
    }
    compile() {
        if (this.block.data != -1) {
            return `dim.runCommand(\`setblock \${loc.x} \${loc.y} \${loc.z} ${this.block.id} ${this.block.data}\`);`;
        }
        else {
            let result = `let block = ctx.minecraftBlockTypes.get('${this.block.id}').createDefaultBlockPermutation();`;
            if (this.block.states) {
                for (const [state, val] of this.block.states.entries()) {
                    result += `\nblock.getProperty('${state}').value = ${typeof val == 'string' ? `'${val}'` : val};`;
                }
            }
            result += '\ndim.getBlock(loc).setPermutation(block);';
            return result;
        }
    }
}
class TypePattern extends PatternNode {
    constructor(token, type) {
        super(token);
        this.type = type;
        this.prec = -1;
        this.opCount = 0;
    }
    compile() {
        let type = this.type;
        if (!type.includes(':')) {
            type = 'minecraft:' + type;
        }
        return `let newBlock = ctx.minecraftBlockTypes.get('${type}').createDefaultBlockPermutation();
let oldBlock = dim.getBlock(loc);
oldBlock.permutation.getAllProperties().forEach(prop => {
    newBlock.getProperty(prop.name).value = prop.value;
})
oldBlock.setPermutation(newBlock);`;
    }
}
class StatePattern extends PatternNode {
    constructor(token, states) {
        super(token);
        this.states = states;
        this.prec = -1;
        this.opCount = 0;
    }
    compile() {
        let result = 'let newBlock = dim.getBlock(loc).permutation.clone();';
        result += '\nlet oldBlock = dim.getBlock(loc);';
        for (const [state, val] of this.states.entries()) {
            result += `\nnewBlock.getProperty('${state}').value = ${typeof val == 'string' ? `'${val}'` : val};`;
        }
        result += '\noldBlock.setPermutation(newBlock);';
        return result;
    }
}
class RandStatePattern extends PatternNode {
    constructor(token, block) {
        super(token);
        this.block = block;
        this.prec = -1;
        this.opCount = 0;
    }
    compile() {
        let type = this.block;
        if (!type.includes(':')) {
            type = 'minecraft:' + type;
        }
        return `let block = ctx.minecraftBlockTypes.get('${type}').createDefaultBlockPermutation();
block.getAllProperties().forEach(state => {
    state.value = state.validValues[Math.floor(Math.random() * state.validValues.length)] ?? state.value;
});
dim.getBlock(loc).setPermutation(block);`;
    }
}
class PercentPattern extends PatternNode {
    constructor(token, percent) {
        super(token);
        this.percent = percent;
        this.prec = 2;
        this.opCount = 1;
    }
    compile() {
        return '';
    }
}
class ChainPattern extends PatternNode {
    constructor() {
        super(...arguments);
        this.prec = 1;
        this.opCount = 2;
        this.evenDistribution = true;
        this.cumWeights = [];
    }
    compile() {
        if (this.nodes.length == 1) {
            return this.nodes[0].compile();
        }
        let result = '';
        let i = 0;
        result += 'let rand = ' + (this.evenDistribution ? `Math.floor(Math.random() * ${this.nodes.length});\n` : `${this.weightTotal} * Math.random();\n`);
        for (const node of this.nodes) {
            if (i != 0) {
                result += 'else ';
            }
            result += `if (${this.evenDistribution ? `rand == ${i}` : `${this.cumWeights[i]} >= rand`}) {(() => {\n${this.nodes[i].compile()}\n})()}\n`;
            i++;
        }
        return result;
    }
    postProcess() {
        super.postProcess();
        const defaultPercent = 100 / this.nodes.length;
        let totalPercent = 0;
        const patterns = this.nodes;
        const weights = [];
        this.nodes = [];
        while (patterns.length) {
            const pattern = patterns.shift();
            if (pattern instanceof ChainPattern) {
                const sub = pattern.nodes.reverse();
                for (const child of sub) {
                    patterns.unshift(child);
                }
            }
            else if (pattern instanceof PercentPattern) {
                this.evenDistribution = false;
                this.nodes.push(pattern.nodes[0]);
                weights.push(pattern.percent);
                pattern.nodes[0].postProcess();
                totalPercent += pattern.percent;
            }
            else {
                this.nodes.push(pattern);
                weights.push(defaultPercent);
                pattern.postProcess();
                totalPercent += defaultPercent;
            }
        }
        weights.map(value => {
            // printDebug(value / totalPercent);
            return value / totalPercent;
        });
        if (!this.evenDistribution) {
            for (let i = 0; i < weights.length; i += 1) {
                this.cumWeights.push(weights[i] + (this.cumWeights[i - 1] || 0));
            }
            this.weightTotal = this.cumWeights[this.cumWeights.length - 1];
        }
    }
}
const patternContext = {
    minecraftBlockTypes: MinecraftBlockTypes,
    placeBlock: function (block, loc, dim) {
        let command = block.id;
        if (block.states && block.states.size != 0) {
            command += '[';
            let i = 0;
            for (const state of block.states.entries()) {
                command += `"${state[0]}":`;
                command += typeof state[1] == 'string' ? `"${state[1]}"` : `${state[1]}`;
                if (i < block.states.size - 1) {
                    command += ',';
                }
                i++;
            }
            command += ']';
        }
        else if (block.data != -1) {
            command += ' ' + block.data;
        }
        dim.runCommand(`setblock ${loc.x} ${loc.y} ${loc.z} ${command}`);
    }
};
